/*
Copyright 2008-2012 Christophe Clmence

This file is part of RobotClic.

RobotClic is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

RobotClic is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with RobotClic. If not, see <http://www.gnu.org/licenses/>.
*/

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.util.*;
import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.nio.channels.FileChannel;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import com.melloware.jintellitype.HotkeyListener;
import com.melloware.jintellitype.JIntellitype;

/**
 * @author Christophe Clmence (Kistrof)
 * Logiciel distribu sous licence GNU GPL v3
 * Vous pouvez le copier et le distribuer librement
 */
public class RobotClic implements KeyListener, FocusListener
{
	private static final String version = "2.7.2";
	private static final String date = "20 aot 2012";
	private static final String website_url = "http://kistrof.free.fr/robotclic/";
	private static final String titre_about = "A propos de RobotClic";
	private static final String titre_update = "Mise  jour";
	private static final String titre_changelog = "Historique des versions";
	private static final String titre_licence = "Licence GNU GPL v3";
	private static final String[] about = {"Version " + RobotClic.version + " - " + RobotClic.date,
		"Copyright 2008-2012 Christophe Clmence",
		"",
		"Dvelopp en Java sous Eclipse 3",
		"",
		"Ce logiciel est distribu sous licence GNU GPL v3",
		"",
		website_url};
	private static final String titre_help = "Aide";
	private static final String[] help = {"[Espace] : ajouter la position de la souris",
		"[Echap] : terminer l'enregistrement des points",
		"[Ctrl] + [Alt] + S : stopper la squence en cours",
		"",
		"Vous trouverez plus d'aide sur",
		website_url};
	public static final String titre_fenetre = "RobotClic v" + RobotClic.version;
	
	public Robot robot;
	public PointClicList liste;
	private Cliqueur cliqueur;
	
	private JFrame preview_frame;
	private PreviewPanel preview;
	
	public JFrame fenetre;
	private JPanel panel;
	
	public JCheckBox replace;
	public JTextField texte;
	
	private JComboBox<PointClic> combobox;
	private JLabel label_tempo, label_ms;
	private JSpinner spin_tempo;
	
	private JButton ok_tempo;
	private TitledBorder bordure_details;
	
	private JPanel panel_liste, panel_details, panel_bas;
	private SpringLayout spring;
	
	private JRadioButton radio_gauche, radio_droit;
	private ButtonGroup radio_group;
	
	private JMenuBar barre;
	private JMenu menu_fichier;
	private JMenuItem file_save, file_load, file_quit;
	private JMenu menu_seq;
	private JMenuItem seq_register, seq_launch, seq_reset;
	private JMenu menu_affichage;
	private JCheckBoxMenuItem aff_apercu;
	private JMenuItem aff_update;
	private JMenu menu_help;
	private JMenuItem help_help, help_about, help_update, help_changelog, help_licence;
	
	public static boolean JIntellitypeEnabled = false;
	
	/**
	 * 
	 * @param args
	 */
	public static void main(String[] args)
	{
		if (System.getProperty("os.name").toLowerCase().contains("windows")) setupJIntellitype();
		try
		{
			new RobotClic();
		}
		catch (AWTException ex)
		{
			JOptionPane.showMessageDialog(null, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
		}
	}
	
	/**
	 * 
	 */
	private static void setupJIntellitype()
	{
		String arch = System.getProperty("sun.arch.data.model");
		File f = new File("JIntellitype.dll");
		if (f.exists()) f.delete();
		FileChannel in = null;
		FileChannel out = null;
		try
		{
			in = new FileInputStream("JIntellitype"+arch+".dll").getChannel();
			out = new FileOutputStream("JIntellitype.dll").getChannel();
			in.transferTo(0, in.size(), out);
			RobotClic.JIntellitypeEnabled = true;
		}
		catch (Exception e) { e.printStackTrace(); }
		finally
		{
			if (in != null) try { in.close(); } catch (IOException e) {}
			if (out != null) try { out.close(); } catch (IOException e) {}
		}
	}
	
	/**
	 * 
	 * @param time
	 */
	private void pause(long time)
	{
		try
		{
			Thread.sleep(time);
		}
		catch (InterruptedException e) {}
	}
	
	/**
	 * 
	 * @throws AWTException
	 */
	private RobotClic() throws AWTException
	{
		try
		{
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		}
		catch (Exception ex)
		{
			JOptionPane.showMessageDialog(null, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
		}
		
		// Robot systme servant  manipuler la souris automatiquement
		robot = new Robot();
		
		// Fentre d'aperu
		preview = new PreviewPanel(robot.createScreenCapture(new Rectangle(java.awt.Toolkit.getDefaultToolkit().getScreenSize())));
        preview_frame = new JFrame("RobotClic - Aperu");
        preview_frame.setContentPane(preview);
        preview_frame.setResizable(false);
        preview_frame.setSize(preview.getWidth()+15, preview.getHeight()+35);
        preview_frame.addWindowListener(new WindowAdapter(){
        	public void windowClosing(WindowEvent e) 
        	{
        		aff_apercu.setState(false);
        	}
        });
		
		
		// Bouton Tous (applique la tempo  tous les points)
		ok_tempo = new JButton("Tous");
		ok_tempo.setEnabled(false);
		ok_tempo.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				if (combobox.getItemCount() > 0)
					appliquerTempoPourTous();
			}
		});
		
		// Liste droulante des points
		combobox = new JComboBox<PointClic>();
		combobox.setEnabled(false);
		combobox.addItemListener(new ItemListener(){
			public void itemStateChanged(ItemEvent e)
			{
				PointClic p_old = (PointClic)e.getItem();
				PointClic p_new = (PointClic)combobox.getSelectedItem();
				if (p_old == p_new)
				{
					spin_tempo.setValue(new Integer(p_new.getTempo()));
					if (p_new.getBouton() == PointClic.droit)
						radio_droit.setSelected(true);
					else
						radio_gauche.setSelected(true);
					dessinerPointCourant();
				}
				else
				{
					preview.reset();
				}
			}
		});
		
		file_save = new JMenuItem("Enregistrer la squence");
		file_save.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				save();
			}
		});
		
		file_load = new JMenuItem("Charger une squence");
		file_load.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				load();
			}
		});
		
		file_quit = new JMenuItem("Quitter");
		file_quit.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				System.exit(0);
			}
		});
		
		seq_register = new JMenuItem("Nouveau");
		seq_register.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				enregistrer();
			}
		});
		
		seq_launch = new JMenuItem("Lancer");
		seq_launch.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				reproduire();
			}
		});
		
		seq_reset = new JMenuItem("Vider");
		seq_reset.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				reset();
			}
		});
		
		aff_apercu = new JCheckBoxMenuItem("Montrer l'aperu");
		aff_apercu.addItemListener(new ItemListener(){
			public void itemStateChanged(ItemEvent e)
			{
				if (e.getStateChange() == ItemEvent.DESELECTED)
					preview_frame.setVisible(false);
				else
					preview_frame.setVisible(true);
			}
		});
		
		aff_update = new JMenuItem("Mettre  jour");
		aff_update.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				boolean visible = preview_frame.isVisible();
				// Cache le prvisualiseur
				enablePreview(false);
				// Cache la fentre principale
				fenetre.setVisible(false);
				// Petite pause pour viter de capturer la fentre principale
				pause(100);
				// Capture l'image
				BufferedImage img = robot.createScreenCapture(new Rectangle(java.awt.Toolkit.getDefaultToolkit().getScreenSize()));
				// Remise en place du prvisualiseur
				enablePreview(visible);
				// Raffiche la fentre principale
				fenetre.setVisible(true);
				// Applique l'image au previsualiseur
				preview.updateBackgroud(img);
				// Redessine le point courant
				dessinerPointCourant();
				// Raffraichit le prvisualiseur
				preview.updateUI();
			}
		});
		
		help_help = new JMenuItem("Aide");
		help_help.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				displayHelp();
			}
		});
		
		help_changelog = new JMenuItem("Nouveauts");
		help_changelog.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				showChangelog();
			}
		});
		
		help_licence = new JMenuItem("Licence");
		help_licence.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				showLicence();
			}
		});
		
		help_update = new JMenuItem("Mise  jour");
		help_update.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				checkUpdate();
			}
		});
		
		help_about = new JMenuItem("A propos");
		help_about.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				displayAbout();
			}
		});
		
		radio_gauche = new JRadioButton("Clic gauche");
		radio_gauche.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				PointClic pc = (PointClic)combobox.getSelectedItem();
				pc.setButton(PointClic.gauche);
				combobox.repaint();
			}
		});
		
		radio_droit = new JRadioButton("Clic droit");
		radio_droit.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e)
			{
				PointClic pc = (PointClic)combobox.getSelectedItem();
				pc.setButton(PointClic.droit);
				combobox.repaint();
			}
		});
		
		radio_group = new ButtonGroup();
		radio_group.add(radio_gauche);
		radio_group.add(radio_droit);
		
		menu_fichier = new JMenu("Fichier");
		menu_fichier.add(file_load);
		menu_fichier.add(file_save);
		menu_fichier.addSeparator();
		menu_fichier.add(file_quit);
		
		menu_seq = new JMenu("Squence");
		menu_seq.add(seq_register);
		menu_seq.add(seq_launch);
		menu_seq.add(seq_reset);
		
		menu_affichage = new JMenu("Aperu");
		menu_affichage.add(aff_apercu);
		menu_affichage.add(aff_update);
		
		menu_help = new JMenu("?");
		menu_help.add(help_help);
		menu_help.add(help_changelog);
		menu_help.add(help_licence);
		menu_help.add(help_update);
		menu_help.add(help_about);
		
		barre = new JMenuBar();
		barre.add(menu_fichier);
		barre.add(menu_seq);
		barre.add(menu_affichage);
		barre.add(menu_help);
		
		label_tempo = new JLabel("Temporisation : ");
		label_ms = new JLabel("ms");
		
		SpinnerNumberModel spinmodel = new SpinnerNumberModel(0, 0, 10, 1);
		spinmodel.setMaximum(null);
		spin_tempo = new JSpinner(spinmodel);
		spin_tempo.setPreferredSize(new Dimension(75, 25));
		spin_tempo.setEnabled(false);
		spin_tempo.addChangeListener(new ChangeListener(){
			public void stateChanged(ChangeEvent e)
			{
				if (combobox.getItemCount() > 0)
					appliquerTempo();
			}
		});
		
		replace = new JCheckBox("Ramener le curseur", true);
		texte = new JTextField();
		texte.setEditable(false);
		texte.setText("");
		
		panel_liste = new JPanel(new BorderLayout());
		panel_liste.add(combobox, BorderLayout.CENTER);
		
		bordure_details = new TitledBorder("Dtails");
		spring = new SpringLayout();
		panel_details = new JPanel(spring);
		panel_details.setBorder(bordure_details);
		panel_details.add(label_tempo);
		panel_details.add(spin_tempo);
		panel_details.add(label_ms);
		panel_details.add(ok_tempo);
		panel_details.add(radio_gauche);
		panel_details.add(radio_droit);
		
		spring.putConstraint(SpringLayout.WEST, label_tempo, 5, SpringLayout.WEST, panel_details);
		spring.putConstraint(SpringLayout.WEST, spin_tempo, 0, SpringLayout.EAST, label_tempo);
		spring.putConstraint(SpringLayout.WEST, label_ms, 5, SpringLayout.EAST, spin_tempo);
		spring.putConstraint(SpringLayout.WEST, ok_tempo, 5, SpringLayout.EAST, label_ms);
		
		spring.putConstraint(SpringLayout.VERTICAL_CENTER, spin_tempo, 0, SpringLayout.VERTICAL_CENTER, label_tempo);
		spring.putConstraint(SpringLayout.VERTICAL_CENTER, label_ms, 0, SpringLayout.VERTICAL_CENTER, spin_tempo);
		spring.putConstraint(SpringLayout.VERTICAL_CENTER, ok_tempo, 0, SpringLayout.VERTICAL_CENTER, label_ms);
		
		spring.putConstraint(SpringLayout.NORTH, radio_gauche, 0, SpringLayout.SOUTH, ok_tempo);
		spring.putConstraint(SpringLayout.NORTH, radio_droit, 0, SpringLayout.SOUTH, radio_gauche);
		spring.putConstraint(SpringLayout.SOUTH, panel_details, 0, SpringLayout.SOUTH, radio_droit);
		spring.putConstraint(SpringLayout.EAST, panel_details, 5, SpringLayout.EAST, ok_tempo);
		
		panel_bas = new JPanel(new BorderLayout());
		panel_bas.add(replace, BorderLayout.NORTH);
		panel_bas.add(texte, BorderLayout.SOUTH);
		
		panel = new JPanel(new BorderLayout());
		panel.add(panel_liste, BorderLayout.NORTH);
		panel.add(panel_details, BorderLayout.CENTER);
		panel.add(panel_bas, BorderLayout.SOUTH);
				
		fenetre = new JFrame(titre_fenetre);
		fenetre.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		fenetre.setContentPane(panel);
		fenetre.setResizable(false);
		fenetre.setJMenuBar(barre);
		
		fenetre.pack();
		fenetre.setLocationRelativeTo(null);
		this.enableInterface(false);
		fenetre.setVisible(true);
	}
	
	/**
	 * Fournit la position courante de la souris
	 * @return Point
	 */
	public Point getMouseLocation()
	{
		PointerInfo pointer = MouseInfo.getPointerInfo();
		Point loc = pointer.getLocation();
		return loc;
	}
	
	/**
	 * Lance la procdure d'enregistrement
	 */
	private void enregistrer()
	{
		// Cache l'aperu
		enablePreview(false);
		// Verrouille l'interface
		this.enableInterface(false);
		// Initialisation de la liste
		liste = new PointClicList();
		// Vidage de la liste droulante
		combobox.removeAllItems();
		// Retire tous les Listeners par prcaution
		this.retirerListeners();
		// Ajout des Listeners sur le champ de texte du bas
		texte.addKeyListener(this);
		texte.addFocusListener(this);
		// Passe le focus sur le champ de texte
		texte.grabFocus();
	}
	
	/**
	 * Parcourt la liste de points et effectue un clic sur chacun.
	 */
	private void reproduire()
	{
		// Cache la fentre
		fenetre.setVisible(false);
		// Dsactive tous les lments de l'interface
		this.enableInterface(false);
		// HotKey global pour stopper la squence
		if (RobotClic.JIntellitypeEnabled)
		{
			JIntellitype.getInstance().addHotKeyListener(new HotkeyListener() {
				public void onHotKey(int aIdentifier) {
					if (aIdentifier == 1 && cliqueur != null)
					{
						cliqueur.interrupt();
					}
				}
			});
			JIntellitype.getInstance().registerHotKey(1, JIntellitype.MOD_CONTROL|JIntellitype.MOD_ALT, (int)'S');
		}
		// Thread de lecture de la squence
		this.cliqueur = new Cliqueur(this);
		this.cliqueur.start();
	}
	
	/**
	 * Fin de la squence : ractivation de l'interface
	 */
	public void endSequence()
	{
		this.cliqueur = null;
		// Ractive les lments de l'interface
		this.enableInterface(true);
		// Raffiche la fentre
		fenetre.setVisible(true);
	}
	
	/**
	 * Efface la liste de points et la liste droulante
	 */
	private void reset()
	{
		liste = new PointClicList();
		this.retirerListeners();
		this.enableInterface(false);
		texte.setText("Liste vide !");
		combobox.removeAllItems();
	}
	
	/**
	 * Raction  l'appui d'une touche du clavier
	 */
	public void keyTyped(KeyEvent e) {}
	public void keyReleased(KeyEvent e) {}
	public void keyPressed(KeyEvent e)
	{
		int code = e.getKeyCode();
		switch (code)
		{
			case KeyEvent.VK_SPACE:
			{
				ajouterPoint();
				break;
			}
			case KeyEvent.VK_ESCAPE:
			{
				finEnregistrement();
				break;
			}
			default:
			{
				break;
			}
		}
	}
	
	/**
	 * Ajoute la position de la souris  la liste
	 */
	private void ajouterPoint()
	{
		Point p = getMouseLocation();
		try
		{
			PointClic pc = new PointClic(p.x, p.y, 0, liste.size()+1);
			liste.add(pc);
			texte.setText("Points ajouts : "+liste.size()+" - Echap pour terminer");
		}
		catch (IOException ex)
		{
			JOptionPane.showMessageDialog(fenetre, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
		}
	}
	
	/**
	 * Retire les listeners (stoppe l'enregistrement)
	 */
	private void finEnregistrement()
	{
		this.retirerListeners();
		if (liste.size() > 0)
			this.enableInterface(true);
		texte.setText("Enregistrement termin, "+liste.size()+" points");
	}
	
	/**
	 * Retire les listeners, dbloque les boutons et actualise la liste droulante
	 */
	private void retirerListeners()
	{
		// KeyListener de texte
		KeyListener[] kl = texte.getKeyListeners();
		for (int i = 0 ; i < kl.length ; i++)
			texte.removeKeyListener(kl[i]);
		// FocusListener de texte
		FocusListener[] fl = texte.getFocusListeners();
		for (int i = 0 ; i < fl.length ; i++)
			texte.removeFocusListener(fl[i]);
		// Actualisation de la liste droulante de points
		combobox.removeAllItems();
		ListIterator<PointClic> it = liste.listIterator();
		while (it.hasNext())
			combobox.addItem(it.next());
	}
	
	/**
	 * Applique la tempo inscrite dans txt_tempo au point courant de la liste droulante
	 */
	private void appliquerTempo()
	{
		Integer i = (Integer)spin_tempo.getValue();
		int t = i.intValue();
		PointClic pc = (PointClic)combobox.getSelectedItem();
		pc.setTempo(t);
		combobox.repaint();
		texte.setText("Tempo point "+pc.getIdent()+" mise  "+t+" ms");
	}
	
	/**
	 * Applique la temporisation t  tous les points
	 * @param t
	 */
	private void appliquerTempoPourTous()
	{
		int t = ((Integer)spin_tempo.getValue()).intValue();
		int nb = combobox.getItemCount();
		for (int i = 0 ; i < nb ; i++)
		{
			PointClic pc = (PointClic)combobox.getItemAt(i);
			pc.setTempo(t);
		}
		combobox.repaint();
		texte.setText("Tempo mises  "+t+" ms");
	}
	
	/**
	 * Active ou dsactive les lments de l'interface
	 * @param active boolean
	 */
	private void enableInterface(boolean active)
	{
		file_save.setEnabled(active);
		seq_launch.setEnabled(active);
		combobox.setEnabled(active);
		spin_tempo.setEnabled(active);
		ok_tempo.setEnabled(active);
		radio_gauche.setEnabled(active);
		radio_droit.setEnabled(active);
		replace.setEnabled(active);
		fenetre.pack();
	}
	
	/**
	 * 
	 * @param active
	 */
	private void enablePreview(boolean active)
	{
		preview_frame.setVisible(active);
		aff_apercu.setState(active);
	}
	
	/**
	 * Ragit au gain du focus
	 */
	public void focusGained(FocusEvent e)
	{
		int s = liste.size();
		if (s == 0)
			texte.setText("[Espace] : nouveau point / [Echap] : finir");
		else
			texte.setText("Points ajouts : "+s+" - Echap pour terminer");
	}
	
	/**
	 * Ragit  la perte du focus
	 */
	public void focusLost(FocusEvent e)
	{
		texte.setText("/!\\ Cliquez ici pour enregistrer /!\\");
	}
	
	/**
	 *
	 */
	private void dessinerPointCourant()
	{
		PointClic p = (PointClic)combobox.getSelectedItem();
		if (p != null)
			preview.dessinerPoint(p);
		preview.updateUI();
	}
	
	/**
	 *
	 */
	private void save()
	{
		JFileChooser chooser = new JFileChooser(new File("."));
		SequenceFileFilter filter = new SequenceFileFilter("Fichiers squence (*.seq)", PointClicList.ext);
		chooser.setFileFilter(filter);
		int returnVal = chooser.showSaveDialog(fenetre);
		if(returnVal == JFileChooser.APPROVE_OPTION)
		{
			File f = chooser.getSelectedFile();
			if (!f.getName().endsWith("."+PointClicList.ext))
			{
				f = new File(f.getAbsolutePath()+"."+PointClicList.ext);
			}
			try
			{
				if (liste != null)
				{
					liste.saveFile(f.getAbsolutePath());
					texte.setText("Enregistrement termin, "+liste.size()+" points");
				}
			}
			catch (IOException ex)
			{
				JOptionPane.showMessageDialog(fenetre, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
			}
		}
		
	}
	
	/**
	 * 
	 */
	private void load()
	{
		JFileChooser chooser = new JFileChooser(new File("."));
		SequenceFileFilter filter = new SequenceFileFilter("Fichiers squence (*.seq)", PointClicList.ext);
		chooser.setFileFilter(filter);
		int returnVal = chooser.showOpenDialog(fenetre);
	    if(returnVal == JFileChooser.APPROVE_OPTION)
	    {
	       File f = chooser.getSelectedFile();
	       try
	       {
	    	   liste = PointClicList.loadFile(f.getAbsolutePath());
	    	   this.retirerListeners();
	    	   texte.setText("Chargement termin, "+liste.size()+" points");
	    	   this.enableInterface(true);
	       }
	       catch (FileNotFoundException ex)
	       {
	    	   JOptionPane.showMessageDialog(fenetre, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
	       }
	       catch (IOException ex)
	       {
	    	   JOptionPane.showMessageDialog(fenetre, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
	       }
	       catch (NumberFormatException ex)
	       {
	    	   JOptionPane.showMessageDialog(fenetre, ex.getMessage(), titre_fenetre, JOptionPane.ERROR_MESSAGE);
	       }
	    }
	}
	
	/**
	 * Affiche l'aide
	 */
	private void displayHelp()
	{
		int nb_msg = RobotClic.help.length;
		JPanel p = new JPanel(new GridLayout(RobotClic.help.length, 1));
		JLabel[] msg_label = new JLabel[nb_msg];
		for (int i = 0 ; i < nb_msg ; i++)
		{
			msg_label[i] = new JLabel(RobotClic.help[i], JLabel.CENTER);
			p.add(msg_label[i]);
		}
		JOptionPane.showMessageDialog(fenetre, p, RobotClic.titre_help, JOptionPane.PLAIN_MESSAGE);
	}
	
	/**
	 * Affiche la fentre "A propos"
	 */
	private void displayAbout()
	{
		int nb_msg = RobotClic.about.length;
		JPanel p = new JPanel(new GridLayout(RobotClic.about.length, 1));
		JLabel[] msg_label = new JLabel[nb_msg];
		for (int i = 0 ; i < nb_msg ; i++)
		{
			msg_label[i] = new JLabel(RobotClic.about[i]);
			p.add(msg_label[i]);
		}
		String[] options = {"Fermer"};
		JOptionPane.showOptionDialog(fenetre, p, RobotClic.titre_about, JOptionPane.OK_OPTION, JOptionPane.INFORMATION_MESSAGE, null, options, options[0]);
	}
	
	/**
	 * Vrifie la prsence d'une mise  jour
	 */
	private void checkUpdate()
	{
		HTTPClient update_checker = new HTTPClient(new String[] {"kistrof.free.fr", "/robotclic/update.php?v="+version});
		update_checker.sendRequest();
		String update_response = new String(update_checker.buf).trim();
		String[] response = {"Aucune mise  jour n'est disponible"};
		if (version.compareTo(update_response) < 0)
		{
			response = new String[2];
			response[0] = "Une mise  jour est disponible : v"+update_response;
			response[1] = "Rendez-vous sur "+website_url+" pour l'obtenir";
		}
		int nb_msg = response.length;
		JPanel p = new JPanel(new GridLayout(response.length, 1));
		JLabel[] msg_label = new JLabel[nb_msg];
		for (int i = 0 ; i < nb_msg ; i++)
		{
			msg_label[i] = new JLabel(response[i], JLabel.CENTER);
			p.add(msg_label[i]);
		}
		JOptionPane.showMessageDialog(fenetre, p, RobotClic.titre_update, JOptionPane.PLAIN_MESSAGE);
	}
	
	/**
	 * Affiche les nouveauts
	 */
	public void showChangelog()
	{
		String texte = lireFichier("doc/changelog.txt");
		if (!texte.equals(""))
		{
			JTextArea t = new JTextArea(texte);
			t.setEditable(false);
			JScrollPane scroll = new JScrollPane(t);
			scroll.setPreferredSize(new Dimension(550, 400));
			JOptionPane.showMessageDialog(fenetre, scroll, RobotClic.titre_changelog, JOptionPane.PLAIN_MESSAGE);
		}
	}
	
	/**
	 * Affiche la licence
	 */
	public void showLicence()
	{
		String texte = lireFichier("doc/licence.txt");
		if (!texte.equals(""))
		{
			JTextArea t = new JTextArea(texte);
			t.setEditable(false);
			JScrollPane scroll = new JScrollPane(t);
			scroll.setPreferredSize(new Dimension(550, 400));
			JOptionPane.showMessageDialog(fenetre, scroll, RobotClic.titre_licence, JOptionPane.PLAIN_MESSAGE);
		}
	}
	
	/**
	 * Lit le fichier donn et retourne l'intgralit de son contenu dans une String
	 * @param fichier String
	 * @return String
	 */
	private String lireFichier(String chemin)
	{
		String ligne, texte = "";
		try
		{
			URL f = this.getClass().getResource(chemin);
			URLConnection url_cnx = f.openConnection();
			InputStreamReader freader = new InputStreamReader(url_cnx.getInputStream(), "ISO-8859-1");
			BufferedReader reader = new BufferedReader(freader);
			while ((ligne = reader.readLine()) != null) texte += ligne+"\n";
		}
		catch (Exception e)
		{
			System.err.println("Erreur : "+e.getMessage());
		}
		return texte;
	}
	
}

class SequenceFileFilter extends javax.swing.filechooser.FileFilter
{
	private String ext;
	private String desc;
	public SequenceFileFilter(String d, String e)
	{
		super();
		desc = d;
		ext = e;
	}
	public boolean accept(File f)
	{
		if (f.isDirectory())
			return true;
		String n = f.getName();
		String e = n.substring(n.lastIndexOf(".")+1);
		if (e.equals(ext))
			return true;
		else
			return false;
	}
	public String getDescription()
	{
		 return desc;
	}
}